import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import ProjectDetailClient, { ProductData } from "./ProjectDetailClient";
import { buildProjectDetailMeta } from "@/lib/seo/projectMeta";

export const dynamic = "force-dynamic";

async function fetchProjectData(courseCode: string, projectCode: string) {
    try {
        const [res1, res2, res3, res4] = await Promise.all([
            API_ENDPOINTS.PROJECTS.DETAIL(courseCode, projectCode),
            API_ENDPOINTS.SUBSECTIONS(10, 79),
            API_ENDPOINTS.SUBSECTIONS(10, 80),
            API_ENDPOINTS.SUBSECTIONS(10, 81),
        ]);

        if (!res1.ok) throw new Error(`HTTP ${res1.status}`);

        // We'll treat the other sections as optional or handle their errors gracefully if needed, 
        // but for now we follow the original logic which likely expects them.

        const [data1, data2, data3, data4] = await Promise.all([
            res1.json(),
            res2.ok ? res2.json() : [],
            res3.ok ? res3.json() : [],
            res4.ok ? res4.json() : [],
        ]);

        return {
            productData: data1.project as ProductData,
            firstSection: data2,
            secondSection: data3,
            thirdSection: data4,
        };
    } catch (err) {
        console.error("❌ Failed to load project details", err);
        return null;
    }
}

export async function generateMetadata({ params }: { params: Promise<{ code: string, projectId: string }> }): Promise<Metadata> {
    const resolvedParams = await params;
    const data = await fetchProjectData(resolvedParams.code, resolvedParams.projectId);

    if (!data || !data.productData) {
        return { title: "Project Detail" };
    }

    return buildProjectDetailMeta(data.productData, resolvedParams.code);
}

export default async function ProjectDetailPage({ params }: { params: Promise<{ code: string, projectId: string }> }) {
    const resolvedParams = await params;
    const data = await fetchProjectData(resolvedParams.code, resolvedParams.projectId);

    if (!data || !data.productData) {
        return <div className="text-danger text-center py-5">Failed to load project details.</div>;
    }

    return (
        <ProjectDetailClient
            productData={data.productData}
            firstSection={data.firstSection}
            secondSection={data.secondSection}
            thirdSection={data.thirdSection}
            courseCode={resolvedParams.code}
            projectCode={resolvedParams.projectId}
        />
    );
}
